/**
   \file veriT-signal.c
   \brief Implementation of signal handling functionality. */

#define _XOPEN_SOURCE
#include "utils/veriT-signal.h"

#include "veriT-config.h"
#include "utils/general.h"
#include "utils/stack.h"
#include "utils/statistics.h"
#include "veriT.h"

#include <signal.h>
#include <stdio.h>
#include <unistd.h>

#ifndef NSIG
#ifdef _NSIG
#define NSIG _NSIG
#else
#define NSIG 32
#endif
#endif

/*
  PF: there are several problems to solve here:
  - one cannot put a handler on SIGSTOP and SIGKILL, and thus I think
  this is not the right way to clear the system from eprover processes
  - I had plenty of problems on grid5k because of SIGSTOP signals while
  profiling.  I think there were generated by the profiler / not sure of it
*/

#ifdef CAPTURE_SIGNALS
TSstack(_sigfun, Tsigfun);

static Tstack_sigfun veriT_sigfun_stack;
#endif

#ifdef CAPTURE_SIGNALS
static void
veriT_signal_handle(int signum)
{
  unsigned i;
  for (i = 0; i < stack_size(veriT_sigfun_stack); i++)
    stack_get(veriT_sigfun_stack, i)();
  switch (signum)
    {
      case SIGINT:
        if (veriT_print_report)
          {
            stats_fprint_formats(stdout);
            stats_fprint(stdout);
            fflush(stdout);
          }
        break;
      case SIGILL: fprintf(stderr, "Illegal Instruction\n"); break;
      case SIGFPE: fprintf(stderr, "Floating point exception\n"); break;
      case SIGSEGV: fprintf(stderr, "Invalid memory reference\n"); break;
      case SIGPIPE: fprintf(stderr, "Broken pipe\n"); break;
      case SIGALRM:
        fprintf(stderr, TIMEOUT_TEXT);
        if (veriT_print_report)
          {
            stats_fprint_formats(stdout);
            stats_fprint(stdout);
            fflush(stdout);
          }
        signum = VERIT_SIGALRM; /* Ensures consistency across systems */
        break;
      case SIGXCPU:
        fprintf(stderr, "CPU time limit exceeded\n");
        if (veriT_print_report)
          {
            stats_fprint_formats(stdout);
            stats_fprint(stdout);
            fflush(stdout);
          }
        signum = VERIT_SIGXCPU; /* Ensures consistency across systems */
        break;
    }
  _exit(signum);
}
#endif

#ifdef CAPTURE_SIGNALS
static struct sigaction new_act, old;
#endif

void
veriT_signal_init(void)
{
#ifdef CAPTURE_SIGNALS
	stack_INIT(veriT_sigfun_stack);
	memset(&new_act, 0, sizeof(new_act)); /* memset necessaire pour flags=0 */
	new_act.sa_handler = veriT_signal_handle;
	sigemptyset(&new_act.sa_mask);
	sigaddset(&new_act.sa_mask, SIGINT);
	sigaddset(&new_act.sa_mask, SIGQUIT);
	sigaddset(&new_act.sa_mask, SIGILL);
	sigaddset(&new_act.sa_mask, SIGABRT);
	sigaddset(&new_act.sa_mask, SIGFPE);
	sigaddset(&new_act.sa_mask, SIGSEGV);
	sigaddset(&new_act.sa_mask, SIGPIPE);
	sigaddset(&new_act.sa_mask, SIGTERM);
	sigaction(SIGINT, &new_act, &old);
	sigaction(SIGQUIT, &new_act, &old);
	sigaction(SIGILL, &new_act, &old);
	sigaction(SIGABRT, &new_act, &old);
	sigaction(SIGFPE, &new_act, &old);
	sigaction(SIGSEGV, &new_act, &old);
	sigaction(SIGPIPE, &new_act, &old);
	sigaction(SIGTERM, &new_act, &old);
	sigaction(SIGXCPU, &new_act, &old);
	sigaction(SIGALRM, &new_act, &old);
#endif
}

void
veriT_signal_done(void)
{
#ifdef CAPTURE_SIGNALS
	new_act.sa_handler = SIG_DFL;
	sigaction(SIGINT, &new_act, &old);
	sigaction(SIGQUIT, &new_act, &old);
	sigaction(SIGILL, &new_act, &old);
	sigaction(SIGABRT, &new_act, &old);
	sigaction(SIGFPE, &new_act, &old);
	sigaction(SIGSEGV, &new_act, &old);
	sigaction(SIGPIPE, &new_act, &old);
	sigaction(SIGTERM, &new_act, &old);
	sigaction(SIGXCPU, &new_act, &old);
	stack_free(veriT_sigfun_stack);
#endif
}

void
veriT_signal_register(Tsigfun sigfun)
{
#ifdef CAPTURE_SIGNALS
	stack_push(veriT_sigfun_stack, sigfun);
#else
#ifdef PEDANTIC
	/* PF just to make sure arguments are used, so that no warning */
	printf("%p\n", sigfun);
#endif
#endif
}
